/** @file   weather.h
 * @brief   Declaration of Weather - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_WEATHER_H
#define H_WWW_WEATHER_H


namespace WeWantWar {


/** @class  Weather
 * @brief   Provides the weather effects to the game.
 * @author  Tomi Lamminsaari
 *
 * The weather effects (rain, thunder) are produced by an instance of this
 * class.
 */
class Weather
{
public:

  ///
  /// Constants, static members and structures
  /// ========================================

  /** Structure for the raindrops
   */
  struct RainDrop {
    /** Altitude of the raindrop ( z-coordinate ). */
    float altitude;
    /** X-coordinate of the raindrop */
    float posX;
    /** Y-coordinate of the raindrop */
    float posY;
  };
  
  /** Bitmask for rain-effects. */
  static const int RAIN = 1;
  /** Bitmask for thunder. */
  static const int THUNDER = 2;
  
  /** The minimum delay between the lightnings (in number of updates) */
  static int MINIMUM_THUNDER_DELAY;
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================
  
  /** Constructor.
   * @param     events            The bitmask that defines, which weather
   *                              effects are in use. Possible weathereffects
   *                              should be or'd together.
   * @see       Public constants, <code>RAIN</code>, and <code>THUNDER</code>
   */
  Weather( int events );
  /** Destructor.
   */
  ~Weather();
  
  
  
  ///
  /// Methods
  /// =======
  
  /** Updates the weather effects */
  void update();
  
  /** Draws the weather effects. */
  void redraw();
  
  
  
protected:

  /** Maximum number of visible raindrops. */
  static const int MAX_RAINDROPS = 70;
  /** The x-coordinate of the raindrops is withing range (-MAX_X/2 ... MAX_X/2) */
  static float MAX_X;
  /** The y-coordinate is withing the range ( -MAX_Y/2 ... MAX_Y/2 ) */
  static float MAX_Y;
  /** Defines how often we adjust the wind's direction and magnitude */
  static int WIND_ADJUST_DELAY;
  /** Maximum magnitude of the wind. */
  static float MAX_WIND_MAGNITUDE;


  /** Initializes the raindrops */
  void initRainDrops();
  
  /** The bitmask for active weather effects. */
  int           m_events;
  /** An array for the raindrops */
  RainDrop*     m_raindrops[ MAX_RAINDROPS ];
  /** Counter that defines the delays of the lightnings */
  int           m_thunderCounter;
  /** When this is set true, the redraw()-method will draw white frame that
   * represents a lightning.
   */
  bool          m_doLightning;
  /** Direction of the wind. 0=up and 256 makes the full circle. */
  int           m_windDirection;
  /** Magnitude of the wind */
  float         m_windMagnitude;
  /** Timer that tells when we should update the direction and magnitude
   * of the wind.
   */
  int           m_windCounter;
  
  
private:
  Weather( const Weather& rW );
  Weather& operator = ( const Weather& rW );
};

};  // end of namespace

#endif // WEATHER_H

/**
 * Version history
 * ===============
 * $Log: weather.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:21+02  lamminsa
 * Initial revision
 *
 */
 
